<?php

namespace App\Http\Controllers;

use App\Models\Event;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class EventController extends Controller
{
    // GET /events (web) - index view
    public function index()
    {
        $events = Event::orderBy('start_at','desc')->paginate(10);
        return view('events.index', compact('events'));
    }

    // Show create form
    public function create()
    {
        return view('events.create');
    }

    // Store new event
    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'start_at' => 'required|date',
            'end_at' => 'required|date|after_or_equal:start_at',
            'photo' => 'nullable|image|max:4096', // 4MB
            'description' => 'nullable|string',
        ]);
 $data['is_published'] = $request->has('is_published');
        if ($request->hasFile('photo')) {
            $path = $request->file('photo')->store('events', 'public');
            $data['photo'] = $path;
        }

        Event::create($data);

        return redirect()->route('events.index')->with('success', 'Event created successfully.');
    }

    // Show single event
    public function show(Event $event)
    {
        return view('events.show', compact('event'));
    }

    // Edit form
    public function edit(Event $event)
    {
        return view('events.edit', compact('event'));
    }

    // Update event
    public function update(Request $request, Event $event)
    {
        $data = $request->validate([
            'name' => 'required|string|max:255',
            'start_at' => 'required|date',
            'end_at' => 'required|date|after_or_equal:start_at',
            'photo' => 'nullable|image|max:4096',
            'description' => 'nullable|string',
        ]);
        $data['is_published'] = $request->has('is_published');

        if ($request->hasFile('photo')) {
            // delete old photo if exists
            if ($event->photo && Storage::disk('public')->exists($event->photo)) {
                Storage::disk('public')->delete($event->photo);
            }
            $path = $request->file('photo')->store('events', 'public');
            $data['photo'] = $path;
        }

        $event->update($data);

        return redirect()->route('events.index')->with('success', 'Event updated successfully.');
    }

    // Delete event
    public function destroy(Event $event)
    {
        if ($event->photo && Storage::disk('public')->exists($event->photo)) {
            Storage::disk('public')->delete($event->photo);
        }
        $event->delete();

        return redirect()->route('events.index')->with('success', 'Event deleted.');
    }

    // API: GET /api/events (returns JSON)
 public function apiIndex(Request $request)
{
    $query = Event::query();

    // 1️⃣ Filter by ID
    if ($request->has('id') && $request->id != '') {
        $query->where('id', $request->id);
    }

    // 2️⃣ Filter by Name (partial search)
    if ($request->has('name') && $request->name != '') {
        $query->where('name', 'LIKE', '%' . $request->name . '%');
    }

    // 3️⃣ Optional upcoming filter
    if ($request->has('upcoming') && $request->upcoming) {
        $query->where('start_at', '>=', now());
    }

    // 4️⃣ Optional Publish / Unpublish Filter
    // Agar query me 'published' pass nahi kiya to sab events return honge
    if ($request->has('published')) {
        if ($request->published === "1" || $request->published === true || $request->published === "true") {
            $query->where('is_published', true);
        } elseif ($request->published === "0" || $request->published === false || $request->published === "false") {
            $query->where('is_published', false);
        }
    }

    $events = $query->orderBy('start_at','asc')->get();

    // Add photo URL + active status as boolean
    $events->transform(function ($e) {
        $e->photo_url = $e->photo ? asset('storage/'.$e->photo) : null;
        $e->active = (bool) $e->is_published; // convert 0/1 to false/true
        return $e;
    });

    return response()->json([
        'success' => true,
        'data' => $events,
    ]);
}

public function togglePublish(Event $event)
{
    $event->is_published = !$event->is_published;
    $event->save();

    return redirect()->back()->with('success', 'Event publish status updated.');
}


}