<?php

namespace App\Http\Controllers;

use App\Models\Blog;
use Illuminate\Http\Request;
use Illuminate\Support\Str; // Import the Str class
use Illuminate\Support\Facades\Storage;


class BlogController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $blogs = Blog::latest()->paginate(10); // Get latest blogs, 10 per page
        return view('blogs.index', compact('blogs'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('blogs.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        // 1. Validate the request
        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp,gif,svg|max:2048',
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_url' => 'nullable|string|max:255',
            'image_alt_tag' => 'nullable|string|max:255',
            // Note: Image handling would be added here
        ]);

          // 2. Handle the file upload
          if ($request->hasFile('image')) {
            // File ko 'storage/app/public/blogs' folder mein save karega
            // aur uska naya unique path return karega
            $imagePath = $request->file('image')->store('blogs', 'public');
            $validatedData['image'] = $imagePath;                                                                                                                                                                                                                                                                                                                                                                                                                                                                                               
        }
        // 2. Create the slug from the title
        $validatedData['slug'] = Str::slug($request->title);

        // 3. Create and save the blog post
        Blog::create($validatedData);

        // 4. Redirect with a success message
        return redirect()->route('blogs.index')->with('success', 'Blog post created successfully!');
    }

    /**
     * Display the specified resource.
     */
    public function show(Blog $blog)
    {
        return view('blogs.show', compact('blog'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Blog $blog)
    {
        return view('blogs.edit', compact('blog'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Blog $blog)
    {
        // 1. Validate the request
        $validatedData = $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,webp,gif,svg|max:2048', // Naya validation
            'meta_title' => 'nullable|string|max:255',
            'meta_description' => 'nullable|string',
            'meta_url' => 'nullable|string|max:255',
            'image_alt_tag' => 'nullable|string|max:255',
        ]);

        // 2. Nayi image ko handle karein
        if ($request->hasFile('image')) {
            // Step A: Purani image ko delete karein (agar hai toh)
            if ($blog->image) {
                Storage::disk('public')->delete($blog->image);
            }

            // Step B: Nayi image ko save karein
            $imagePath = $request->file('image')->store('blogs', 'public');
            $validatedData['image'] = $imagePath;
        }

        // 2. Update the slug if the title has changed
        $validatedData['slug'] = Str::slug($request->title);

        // 3. Update the blog post
        $blog->update($validatedData);

        // 4. Redirect with a success message
        return redirect()->route('blogs.index')->with('success', 'Blog post updated successfully!');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Blog $blog)
    {
        $blog->delete();
        return redirect()->route('blogs.index')->with('success', 'Blog post deleted successfully!');
    }

   
}